<?php
/*--------------------------------------------------------------------------------------------------
    AbstractValidatableObject.php 2020-03-18
    Gambio GmbH
    http://www.gambio.de
    Copyright (c) 2016 Gambio GmbH
    Released under the GNU General Public License (Version 2)
    [http://www.gnu.org/licenses/gpl-2.0.html]
    --------------------------------------------------------------------------------------------------
 */

namespace Gambio\Shop\SellingUnit\Unit\ValueObjects;


use Exception;

abstract class AbstractValidatableObject
{
    /**
     * @var ?Exception
     */
    protected $exception;

    /**
     * @return Exception|null
     */
    public function exception(): ?Exception
    {
        return $this->exception;
    }

    /**
     * @param Exception $exception
     * @throws \ReflectionException
     */
    protected function flattenExceptionBacktrace(\Exception $exception): void
    {
        $traceProperty = (new \ReflectionClass('Exception'))->getProperty('trace');
        $traceProperty->setAccessible(true);

        $flatten = static function (&$value, $key) {
            if ($value instanceof \Closure) {
                $closureReflection = new \ReflectionFunction($value);
                $value = sprintf(
                    '(Closure at %s:%s)',
                    $closureReflection->getFileName(),
                    $closureReflection->getStartLine()
                );
            } elseif (is_object($value)) {
                $value = sprintf('object(%s)', get_class($value));
            } elseif (is_resource($value)) {
                $value = sprintf('resource(%s)', get_resource_type($value));
            }
        };

        do {
            $trace = $traceProperty->getValue($exception);
            foreach ($trace as &$call) {
            
                if (null !== $call['args']) {
                    
                    array_walk_recursive($call['args'], $flatten);
                }
            }
            unset($call);
            $traceProperty->setValue($exception, $trace);
        } while ($exception = $exception->getPrevious());

        $traceProperty->setAccessible(false);
    }

    /**
     * @param Exception $exception
     * @throws \ReflectionException
     */
    public function stackException(\Exception $exception): void
    {
        if ($exception->getPrevious() !== $this->exception) {
            throw new \InvalidArgumentException("StackedException doesn't follow the hierarchy");
        }
        $this->flattenExceptionBacktrace($exception);
        $this->exception = $exception;

    }

}